# -*- coding: utf-8 -*-
"""
Created on Wed Feb 26 10:50:00 2025

@author: Bernd Marcus

 *** Revision ***
 - 20.08.25: originale "flstkennz" werden nicht mehr in "st_alkis" importiert

"""

import sys
from os import path, mkdir
from time import sleep
from datetime import datetime
import sqlite3
import pandas as pd
import requests
from lxml import etree

#   gdal/ogr
from osgeo import ogr

#   QGIS
from qgis.utils import spatialite_connect
from qgis.core import QgsApplication

#   Qt
from PyQt5.QtWidgets import (
    QApplication, QFileDialog, QMessageBox, QDialog,
    QLabel, QLineEdit, QComboBox, QPushButton, QToolButton,
    QVBoxLayout, QHBoxLayout,
    )

class UiWfsProcessor(QDialog):
    def __init__(self):  # , ac_conf):
        super().__init__()
        self.liegeo_path = ''
        self.ac_conf = ''

        self.major_days_difference = None

        # self.selected_combo_box_index = None
        # self.df = None

        self.alkis_container_path()

    def alkis_container_path(self):

        saved_path = loadPathFromConf(conf_file_path)

        if saved_path:  # check: saved_path not None 
            if checkPath(saved_path):
                self.liegeo_path = saved_path
                # print('Pfad ist gültig')
            else:
                print('Pfad ist ungültig')

        else:
            print('Datei nicht gefunden')

        try:
            with sqlite3.connect(self.liegeo_path) as conn:
                df = pd.read_sql_query(
                    '''
                    select datei
                    from t_sys_pfad
                    where name == 'ALKIS-Container';
                    '''
                    , conn
                )
            conn.close()

            self.ac_path = df.loc[0, 'datei']
            print(f'ac_path = {self.ac_path}')

            self.getUpdateDate()

        except sqlite3.Error as e:
            print(f'Die ausgewählte Datei ist keine LieGeo-Datenbank!\n{e}')


    def getUpdateDate(self):
        with sqlite3.connect(self.ac_path) as conn:
            df = pd.read_sql_query(
                '''
                select * from t_sys_conf;
                '''
                , conn
            )
        conn.close()

        # Zeitdifferenz
        #   aktuelles Datum
        current_date = datetime.now()
        #   vergangene Tage letztes Haupt-Updates
        major_date = df.loc[0, 'last_major_alkis_update']
        major_iso_date = datetime.fromisoformat(major_date[:10])  # major_date.replace('Z', '+00:00'))
        major_time_difference = current_date - major_iso_date
        self.major_days_difference = major_time_difference.days
        #   vergangene Tage letztes Teil-Updates
        minor_date = df.loc[0, 'last_minor_alkis_update']
        minor_iso_date = datetime.fromisoformat(minor_date[:10])  # minor_date.replace('Z', '+00:00'))
        minor_time_difference = current_date - minor_iso_date
        self.minor_days_difference = minor_time_difference.days
        # Pfad LieGeo-Db
        # self.liegeo_path = df.loc[0, 'pfad_liegeo']

        # Aufruf UI
        self.initUI()


    def initUI(self):
        lgdb_name = 'LieGeo-Datenbank'

        layout = QVBoxLayout()
        layout.setSpacing(20)  # Größerer Abstand zwischen den Pfadangaben

        self.ok_button = QPushButton("OK")
        self.ok_button.setEnabled(False)
        self.ok_button.clicked.connect(self.on_ok_clicked)

        cancel_button = QPushButton("Abbrechen")
        cancel_button.clicked.connect(self.close)

        vbox = QVBoxLayout()
        vbox.setSpacing(5)  # Abstand zwischen Label und Textfeld auf 5px setzen

        label = QLabel(lgdb_name)
        vbox.addWidget(label)

        hbox = QHBoxLayout()

        line_edit = QLineEdit(self.liegeo_path)
        line_edit.setObjectName(lgdb_name)
        line_edit.setFixedWidth(600)  # Breite auf 600px setzen
        line_edit.setReadOnly(True)

        hbox.addWidget(line_edit)

        button = QToolButton()
        button.setText("...")
        button.clicked.connect(lambda _, le=line_edit, k=lgdb_name: self.open_file_dialog(le, k))
        hbox.addWidget(button)

        vbox.addLayout(hbox)
        layout.addLayout(vbox)

        # Kombobox hinzufügen
        self.combo_box = QComboBox()
        # self.combo_box.addItem("Bitte auswählen")
        self.combo_box.addItems(
            [
                " - Auswahl treffen -",
                "alle Flurstücke aktualisieren",
                "nur neue Flurstücke aktualisieren"
            ])
        self.combo_box.currentIndexChanged.connect(self.on_combobox_changed)
        layout.addWidget(self.combo_box)



        button_layout = QHBoxLayout()
        button_layout.addWidget(self.ok_button)
        button_layout.addWidget(cancel_button)
        layout.addLayout(button_layout)

        self.setLayout(layout)
        self.setWindowTitle('WFS-Download')
        self.show()

        # Initiale Überprüfung der Pfade und der Kombobox nach der vollständigen Initialisierung des UI-Layouts
        self.check_valid()

        self.exec_()

    def open_file_dialog(self, line_edit, lgdb_name):
        file_path = get_file_path(self, f"{lgdb_name} auswählen", "LieGeo-Db (liegeo.sqlite)")
        if file_path:
            line_edit.setText(file_path)
            if file_path != self.liegeo_path:  # Überprüfen, ob sich die Pfade unterscheiden
                self.liegeo_path = file_path
                if savePathToConf(self.liegeo_path, conf_file_path):  # Aktualisiere nur, wenn nötig
                    print(f"LieGeo Pfad geändert und in Konfigurationsdatei aktualisiert: {self.liegeo_path}")
                else:
                    print("Fehler beim Aktualisieren der Konfigurationsdatei.")
            else:
                print("Pfad nicht geändert.")

        self.check_valid()


    def check_valid(self):
        valid_item = self.combo_box.currentIndex() != 0
        self.ok_button.setEnabled(valid_item)


    def on_combobox_changed(self):
        self.check_valid()

    def on_ok_clicked(self):
        # print("OK clicked")
        self.selected_combo_box_index = self.combo_box.currentIndex()
        self.close()


# %% [WFS-Download] ------------------------------------------------------------

class WfsProcessor:

    def __init__(self):
        # Weiche für Paging
        self.is_paging = False
        self.anzahl_members = 0


    def gml2sqlite(self, df_requests):
        # Gesamtzahl berechnen
        total_requests = len(df_requests)

        # Schleifendurchlauf mit Index
        for i, row in enumerate(df_requests.itertuples(), 1):  # Index-Start=1
            print(f'Anfrage an {row.gema_nr}: Land: {row.bundesland} '
                + f'| LK: {row.landkreis} | Gem: {row.gemeinde} '
                + f'| Gema: {row.gemarkung} ({i}/{total_requests})')

            self.sqlite2spatialite(row)


    def sqlite2spatialite(self, row, startindex=0):
        file_tmp_gml = path.normpath(path.join(dir_proj_data, 'wfs_request.gml'))

        try:
            indexed_request = (row.request + '&STARTINDEX=' + str(startindex))
                # debugging
                # + '&COUNT=3000')
            
            response = requests.get(indexed_request)
            response.raise_for_status()  # Prüft HTTP-Fehler (4xx oder 5xx)
            xml_content = response.content

            # Anzahl zurückgegebener Elemente
            root = etree.fromstring(xml_content)

            namespaces = {
                "wfs": "http://www.opengis.net/wfs/2.0",
                # "gml": "http://www.opengis.net/gml/3.2",
                # "ave": "http://repository.gdi-de.org/schemas/adv/produkt/alkis-vereinfacht/2.0"
                }

            member_elements = root.xpath("//wfs:member", namespaces=namespaces)
            self.anzahl_members = len(member_elements)
                        
            print(f'\t{self.anzahl_members} downloadbare Flurstücke gefunden.')

            if self.anzahl_members == 0:
                print(f'\tDer Gemarkungsschlüssel {row.gema_nr} existiert nicht!')

            with open(file_tmp_gml, 'wb') as f:
                f.write(xml_content)

            driver = ogr.GetDriverByName('GML')
            # data_source = driver.Open(file_tmp_gml, 0)
            with driver.Open(file_tmp_gml, 0) as data_source:

                sleep(0.5)

                if data_source is None:  # Überprüfung auf erfolgreiches Öffnen der Datenquelle
                    raise Exception("GML-Datei konnte nicht geöffnet werden.")

                with sqlite3.connect(db_ac) as conn:
                    cur = conn.cursor()
                    layer = data_source.GetLayer()
                    
                    
                    # ####
                    
                    # daLayer = dataSource.GetLayer()
                    # layerDefinition = daLayer.GetLayerDefn()


                    # for i in range(layerDefinition.GetFieldCount()):
                    #     print(layerDefinition.GetFieldDefn(i).GetName())
                    
                    # ###
                    

                    for feature in layer:
                        geom = feature.GetGeometryRef()
                        flstkennz = feature.GetField(row.flstkennz_attr)

                        # Vereinfachte Flurstückskennzeichen-Formatierung
                        flstkennz = (str(flstkennz).ljust(20, '_')
                            if row.gema_nr.startswith('11')
                            else str(flstkennz).rjust(20, '0')
                            )

                        flaeche_amt = feature.GetField(row.flaeche_attr)
                        aktualitaet = feature.GetField(row.aktualitaet_attr)

                        sql = (
                            f'''
                            INSERT INTO {tbl_wfs_wkt}(
                                flstkennz, flaeche_amt, aktualitaet, epsg, wkt
                            )
                            VALUES (?,?,?,?,?)
                            '''
                            )
                        cur.execute(
                            sql, (flstkennz, flaeche_amt, aktualitaet, row.epsg, geom.ExportToWkt())
                            )

                    conn.commit()  # Commit außerhalb der Schleife, aber innerhalb des with-Blocks

        except requests.exceptions.RequestException as e:
            self.log_error(file_err_req, f"Fehler bei der Anfrage für {row.gema_nr}: {e}\nRequest: {row.request}")
            print(f"\tFehler beim Abrufen der URL: {e}")
            print(f"\tGemarkung: {row.gema_nr} - REQUEST GESCHEITERT! Siehe Log-Datei.")
            # if response is not None:
            #     print(f"Response-Text: {response.text}")
        except etree.XMLSyntaxError as e:
            print(f"\tFehler beim Parsen des XML: {e}")
            if response is not None:
                print(f"Response-Text: {response.text}")
        except Exception as e:  # Allgemeiner Fehler-Handler für alle anderen Fehler
            self.log_error(file_err_req, f"Fehler bei der Verarbeitung von {row.gema_nr}: {e}\nRequest: {row.request}\nResponse: {xml_content if 'xml_content' in locals() else 'Nicht verfügbar'}") # Bedingte Ausgabe von xml_content
            with open(file_err_gema, 'a') as f:
                f.write(f"{row.gema_nr}\n")

            print(f'\tFehlermeldung: {e}')
            print('\tVERARBEITUNG GESCHEITERT! Siehe Log-Datei.')

        finally:
            if 'conn' in locals(): #Sicherstellen, dass die Verbindung existiert
                conn.close()

            if pd.notna(row.count_default):
                self.int_count_default = int(row.count_default)
                if self.anzahl_members == self.int_count_default:
                    if not self.is_paging:
                        print('Anzahl der Flurstücke entspricht dem '
                            +  'Download-Limit des WFS-Dienstes.'
                            + '\nNeue Download-Sequenz wird gestartet.'
                            )
                        self.is_paging = True
                        self.paging(row)

    def paging(self, row):
        # int_count_default = int(row.count_default)

        i = 1  # Initialisierung von i
        while self.anzahl_members >= self.int_count_default:
            start_index = self.int_count_default * i
            print(f'{i+1}. Sequenz des Downloads:\n\tStartindex = {start_index}')
            self.sqlite2spatialite(row, startindex=start_index)
            i += 1

        # Zurücksetzen von is_paging
        self.is_paging = False

    # Log schreiben
    def log_error(self, file_path, message):
        timestamp = datetime.now().strftime('%H:%M:%S')
        with open(file_path, 'a') as f:
            f.write(f"-----\n{timestamp}:\n{message}\n")


# %% [Programm-Funktionen] -----------------------------------------------------

# Dialog Dateiauswahl
def get_file_path(parent=None, title="Datei auswählen", filter=""):
    """Öffnet einen QtFileDialog und gibt den ausgewählten Pfad zurück."""
    options = QFileDialog.Options()
    file_path, _ = QFileDialog.getOpenFileName(
        parent, title, "", filter, options=options
        )
    return file_path

# Auswahl LieGeo-Db
def getLiegeoPath():
    """Öffnet Dialog zur Auswahl der LieGeo-DB und gibt den Pfad zurück."""
    return get_file_path(  # Parent, Titel, Filter
        None,
        "LieGeo-Datenbank auswählen",
        "LieGeo-Datenbanken (*liegeo.sqlite)"
        )

# LieGeo-Pfad in Datei speichern
def savePathToConf(conf_path, conf_file):
    """Speichert den Pfad in einer Textdatei."""
    try:
        with open(conf_file, 'w') as f:
            f.write(conf_path)
        return True
    except Exception as e:
        print(f"Pfad konnte nicht gespeichert werden:\n{e}")
        return False

# LieGeo-Pfad auslesen
def loadPathFromConf(conf_file):
    """Lädt den Pfad aus der Textdatei."""
    try:
        with open(conf_file, 'r') as f:
            return f.read().strip()
    except FileNotFoundError:
        return None

# LieGeo-Pfad auf Gültigkeit prüfen
def checkPath(liegeo_path):
    """Überprüft, ob der Pfad gültig ist."""
    return path.exists(liegeo_path)

# Hinweis
def show_message(message, title="Hinweis"):
    """Zeigt eine Meldung in einer MessageBox an."""
    QMessageBox.information(None, title, message)

def main():
    if saved_path and checkPath(saved_path):
        # show_message(f"Gespeicherter LieGeo-Pfad: \n{saved_path}")
        pass
    else:
        new_path = getLiegeoPath()
        if new_path:
            if savePathToConf(new_path, conf_file_path):
                show_message(f"Pfad erfolgreich gespeichert: {new_path}")
            else:
                show_message("Fehler beim Speichern des Pfads.", title="Fehler")
        else:
            show_message("Kein Pfad ausgewählt.", title="Hinweis")


# %% [Programm-Aufruf]

if QApplication.instance() is None:
    _ = QApplication(sys.argv)  # Spyder
else:
    _ = QApplication.instance()  # QGIS

proj_path = QgsProject.instance().absolutePath()
# !!!proj_path = 'C:\LieMaS_Geo_dev\liegeo_admin_tools\liegeo__alkis_container'
conf_file = 'ALKIS_Container.conf'
conf_file_path = path.normpath(path.join(proj_path, conf_file))

saved_path = loadPathFromConf(conf_file_path)

main()
inst_UiWfsProcessor = UiWfsProcessor()
switch = None

if hasattr(inst_UiWfsProcessor, 'selected_combo_box_index'):
    switch = inst_UiWfsProcessor.selected_combo_box_index
else:
    pass

if switch in (1,2):

    globaltime_1 = datetime.now()
    time_1a = datetime.now()

    #-------------------------------------------------------------------------------
    # ANPASSEN
    # Weiche: alle / nur neue Flurstücke
    #   über type_of_wfs_update.combo_box.currentIndex()
    #   aus "wfs__Download_Flurstuecke"
    #   mit: alle=1, neue=2
    # switch = 1
    # switch = type_of_wfs_update.combo_box.currentIndex()
    #-------------------------------------------------------------------------------

    # Exceptions sind im Modul "ogr" standartmäßig deaktiviert,
    # dennoch erscheint Fehlermeldung;
    # ergo: unterdrücken
    ogr.UseExceptions()
    # ogr.DontUseExceptions()

    # ----- Pfade
    # LieGeo
    db_liegeo = inst_UiWfsProcessor.liegeo_path
    # ALKIS-Container
    db_ac = inst_UiWfsProcessor.ac_path

    # Stammverzeichnis Liegeo-Db
    dir_proj_data = path.dirname(db_liegeo)
    # Großelternverzeichnis
    dir_proj = path.dirname(dir_proj_data)

    # ----- Datenbank Tabellen
    #   tbl: Geometrie als WKT
    tbl_wfs_wkt = 't_alkis__wfs_wkt'
    #   view: WKT konvertiert zu Geom
    #view_wfs = 'v_alkis__wfs'

    # ----- Log-Files
    log = '/log'
    dir_ac_log = path.normpath(proj_path + log)
    if path.exists(dir_ac_log):
        pass
    else:
        # mkdir(dir_proj + log)
        mkdir(dir_ac_log)

    # Log-File fehlerhafter Response
    file_err_req = path.normpath(dir_ac_log + '/wfs_request_error.log')
    # Log-File fehlender Gemarkungen
    file_err_gema = path.normpath(dir_ac_log + '/wfs_missing_gema.log')

    # Log-Files leeren
    with open(file_err_req, 'w') as file:
        pass
    with open(file_err_gema, 'w') as file:
        pass
    print('Log-Dateien geleert')

    time_1z = datetime.now()
    print('Initialisierungsdauer: ' + str(time_1z-time_1a))


# %% [Bereitstellung Gemarkungsliste für WFS-Dienste]

    time_2a = datetime.now()

    # SQL
    # LieGeo - Gemarkungsliste:: --------------------------------------------------
    with sqlite3.connect(db_liegeo) as conn:

        # DataFrame ['land_nr', 'gema']
        df_gema_lg = pd.read_sql_query('''
            with
            cte1 as (
                select ROW_NUMBER() over (
                    PARTITION BY land_nr order by gema_nr desc
                    ) as row_num
                    , land_nr, gema_nr
                    -- Fehlstellen abfangen
                    , coalesce(landkreis,'-') as landkreis
                    , coalesce(gemeinde, '-') as gemeinde
                    , coalesce(gemarkung, '-') as gemarkung
                from v_sys_flstkennz
                group by gema_nr
            )
            SELECT * from cte1
            where true
            -- Test:
            ---  Beschränkung auf n Gemarkungen pro Land
            --and row_num <=1
            ---  Gemarkung
            --and gema_nr in ('036372','036377','031903','030903','031901','032167','033413','036271')
            --and gema_nr == '031901'
            --and gema_nr == '011034'
            ;
            '''
            , conn
        )
    conn.close()

    # ALKIS-Container - Gemarkungsliste:: -----------------------------------------
    with sqlite3.connect(db_ac) as conn:

        # DataFrame ['land_nr', 'gema']
        df_gema_ac = pd.read_sql_query('''
            select gema_nr
            from v_gema
            ;
            '''
            , conn
            # Index
            # , index_col='land_nr'
        )
    conn.close()

    # df_gema_ac.gema_nr not in df_gema_lg.gema_nr
    #   1. Negation
    df_gema_ac_not_in_df_gema_lg = df_gema_lg[~df_gema_lg['gema_nr'].isin(
        df_gema_ac['gema_nr'])]

    #   2. Merge
    # df_gema_ac_not_in_df_gema_lg = df_gema_lg.merge(df_gema_ac, on='gema_nr', how='left'
    #                                           , indicator='Quelle')
    # df_gema_ac_not_in_df_gema_lg = df_gema_ac_not_in_df_gema_lg[
    #     df_gema_ac_not_in_df_gema_lg['Quelle'] == 'left_only']

    # ALKIS-Container - WFS-Dienste der Länder:: ----------------------------------
    with sqlite3.connect(db_ac) as conn:

        # DataFrame ['land_nr', 'url', 'typename', 'epsg' ]
        df_wfs = pd.read_sql_query('''
            select pk, bundesland, epsg
                , request_get_capabilities, request_get_feature
                , filter_start, filter_end
                , flstkennz_attr, flaeche_attr, aktualitaet_attr
            from v_sys_wfs
            ;
            '''
            , conn
            # Index
            # , index_col='pk'

        )
    conn.close()


    # %% [getCapa]
    # WFS - GetCapabilities:: -----------------------------------------------------
    # Bundesländer mit niederschwelligem Paging
    # 2025-08-05: 'NI' entfernt
    low_count_default = ['SL','MV','SN','TH']

    df_low_count_default = df_wfs[
        df_wfs['bundesland'].isin(low_count_default)
        ]

    df_request_get_capabilities = pd.DataFrame(
        {
          'request': df_low_count_default['request_get_capabilities']
          , 'bundesland': df_low_count_default['bundesland']
          })

    def get_count_default(row):
        try:
            response = requests.get(row['request'])
            response.raise_for_status()
            root = etree.fromstring(response.content)
            namespaces = {'ows': 'http://www.opengis.net/ows/1.1'}
            return root.find('.//ows:Constraint[@name="CountDefault"]/ows:DefaultValue', namespaces).text

        except requests.exceptions.RequestException as e:
            print(f"Fehler bei der Anfrage für {row['bundesland']}: {e}")
            return None  # Fehlerwert zurückgeben
        except AttributeError as e:
            print(f"Fehler beim Parsen von CountDefault für {row['bundesland']}: {e}")
            return None  # Fehlerwert zurückgeben
            # return 10000  # feste Anzahl, falls CountDefault is None
        except Exception as e:
            print(f"Unbekannter Fehler für {row['bundesland']}: {e}")
            return None  # Fehlerwert zurückgeben

    df_request_get_capabilities['count_default'] = df_request_get_capabilities.apply(get_count_default, axis=1)
    # print(df_request_get_capabilities)

    df_wfs = pd.merge(df_wfs, df_request_get_capabilities[
        ['count_default', 'bundesland']], on='bundesland', how='left')

    # %% [getFeat]
    # WFS - GetFeature:: ----------------------------------------------------------
    if switch == 1:
        df_wfs_admin = pd.merge(df_wfs, df_gema_lg, how='inner'
        , left_on='pk', right_on='land_nr')

        sql_download_date = (
            '''
            update t_sys_conf
            set last_major_alkis_update = date('now')
            , last_minor_alkis_update = date('now')
            ;'''
            )

    elif switch == 2:
        df_wfs_admin = pd.merge(df_wfs, df_gema_ac_not_in_df_gema_lg, how='inner'
                            , left_on='pk', right_on='land_nr')

        sql_download_date = (
            '''
            update t_sys_conf
            set last_major_alkis_update = date('now')
            ;'''
            )

    df_request_get_feature = pd.DataFrame(
        {
         'request': df_wfs_admin['request_get_feature']
             + df_wfs_admin['filter_start']
             + df_wfs_admin['gema_nr']
             + df_wfs_admin['filter_end']
         , 'epsg': df_wfs_admin['epsg']
         , 'count_default': df_wfs_admin['count_default']
         , 'bundesland': df_wfs_admin['bundesland']
         , 'landkreis': df_wfs_admin['landkreis']
         , 'gemeinde': df_wfs_admin['gemeinde']
         , 'gemarkung': df_wfs_admin['gemarkung']
         , 'gema_nr': df_wfs_admin['gema_nr']
         , 'flstkennz_attr': df_wfs_admin['flstkennz_attr']
         , 'flaeche_attr': df_wfs_admin['flaeche_attr']
         , 'aktualitaet_attr': df_wfs_admin['aktualitaet_attr']
         })


    time_2z = datetime.now()
    print('Bereitstellung von DataFrames: ' + str(time_2z - time_2a))

    # Instanz der Klasse
    inst_processor = WfsProcessor()
    # Aufruf Instanzmethode
    inst_processor.gml2sqlite(df_request_get_feature)

    # show_message("Erfolgreicher Download der Flurstücke", title="Download-Bestätigung")

    # %% [Migration nach SpatiaLite]
    sleep(0.5)

    # Spatialite Treiber
    conn = spatialite_connect(db_ac)

    print('Aktualisierung: Flurstücke')
    try:
        with conn:
            # Cursor-Objekt muss nicht explizit angegeben werden
            # https://docs.python.org/3/library/sqlite3.html#how-to-use-connection-shortcut-methods
            # cur = conn.cursor()
            _ = conn.executescript('''
                BEGIN;
                -- UPSERT: Aktualisieren von st_alkis
                INSERT INTO st_alkis (
                    -- flstkennz, fk_flstkennz_liemas, flaeche_amt
                    -- Änderung 20.08.25:
                    fk_flstkennz_liemas, flaeche_amt
                    , gueltig_von , relation
                    , geom
                )
                    -- select flstkennz, fk_flstkennz_liemas, flaeche_amt
                    select fk_flstkennz_liemas, flaeche_amt
                        --, date('now')
                        , aktualitaet
                        , 'Flurstück [OpenData]'
                        -- Reprojektion
                        , st_transform(geom, 3035)
                    from v_alkis__wfs
                    WHERE true
                ON CONFLICT(fk_flstkennz_liemas)
                DO UPDATE
                -- Änderung 20.08.25:
                -- SET flstkennz = excluded.flstkennz
                -- , flaeche_amt = excluded.flaeche_amt
                SET flaeche_amt = excluded.flaeche_amt
                , gueltig_von = excluded.gueltig_von
                , relation = excluded.relation
                -- Änderung 26.08.25 ("alkis_import" Default: CURRENT_DATE):
                , alkis_import = date('now')
                , geom = excluded.geom
                -- Änderung 22.08.25:
                -- WHERE excluded.geom != st_alkis.geom
                -- Änderung 27.08.25:
                -- WHERE excluded.gueltig_von >= st_alkis.gueltig_von
                WHERE coalesce(excluded.gueltig_von, date(0)) >= coalesce(st_alkis.gueltig_von, date(0))
                -- Ausschluss Gemarkungen
                --and length(st_alkis.flstkennz) != 6
                -- Änderung 20.08.25:
                and length(st_alkis.fk_flstkennz_liemas) != 6
                ;
                COMMIT;
            ''')
            # Transaktion wird explizit im SQL-Skript abgeschlossen
            # https://docs.python.org/3/library/sqlite3.html#sqlite3.Cursor.executescript
            # conn.commit()

        print('Flurstücke wurden aktualisiert')

    except sqlite3.Error as e:
        print('Fehlercode: ' + str(e))


    # Aktualisierung: "major_date", "minor_date"
    try:
        with conn:
            _ = conn.execute(sql_download_date)

    except sqlite3.Error as e:
        print('Fehlercode: ' + str(e))


    print('Aktualisierung: Gemarkungen')
    
    try:
        with conn:
            _ = conn.executescript('''
                BEGIN;
                -- UPSERT: Gemarkungen berechnen und einfügen
                INSERT INTO st_alkis (
                    --flstkennz
                    -- Änderung 20.08.25:
                    -- fk_flstkennz_liemas, gueltig_von, relation
                    -- Änderung 26.08.25:
                    fk_flstkennz_liemas, alkis_import, relation
                    , geom
                )
                -- Gemarkungsumringung
                select substr(fk_flstkennz_liemas, 1, 6)
                    , date('now')
                    , 'Gemarkung'
                    , castToMultipolygon(
                        st_buffer(
                            st_boundary(st_union(st_transform(geom, 3035)))
                            , 0.001
                        )
                    )
                from v_alkis__wfs
                WHERE true
                group by substr(fk_flstkennz_liemas, 1, 6)
                -- ON CONFLICT(flstkennz)
                -- DO UPDATE
                -- SET geom = excluded.geom
                -- WHERE excluded.geom != st_alkis.geom
                -- and length(st_alkis.flstkennz) == 6
                -- Änderung 20.08.25:
                ON CONFLICT(fk_flstkennz_liemas)
                DO UPDATE
                -- Änderung 26.08.25 ("alkis_import" Default: CURRENT_DATE):
                SET alkis_import = date('now')
                , geom = excluded.geom
                WHERE excluded.geom != st_alkis.geom
                and length(st_alkis.fk_flstkennz_liemas) == 6
                ;

                COMMIT;
            ''')

        print('Gemarkungen wurden aktualisiert')

    except sqlite3.Error as e:
        print('Fehlercode: ' + str(e))


    print('Aktualisierung: Datumstempel von Flurstücken mit veraltertem Folgemerkmal')
    
    try:
        with conn:
            _ = conn.executescript('''
                BEGIN;
                update st_alkis
                set gueltig_bis = coalesce(date(a.datum, '-1 day'), date('now','localtime')) from (
                    with
                    cte1 as (
                        select pk, fk_flstkennz_liemas
                        from st_alkis
                        where substr(fk_flstkennz_liemas,19,2) != '__'
                        and length(fk_flstkennz_liemas) == 20
                    )
                    , cte2 as (
                        select ROW_NUMBER() OVER (
                            PARTITION BY substr(fk_flstkennz_liemas, 1,18)
                            ORDER BY iif(
                                substr(fk_flstkennz_liemas, 19,2) == '__'
                                , substr(fk_flstkennz_liemas, 1, 18) || '00'
                                , fk_flstkennz_liemas
                            ) desc
                            ) AS rang
                        , pk, fk_flstkennz_liemas
                        , substr(gueltig_von,1,4) ||'-'|| substr(gueltig_von,6,2) ||'-'|| substr(gueltig_von,9,2) as datum
                        from st_alkis
                        where 1
                        and substr(fk_flstkennz_liemas, 1, 18) in (
                            select substr(fk_flstkennz_liemas, 1, 18) from cte1
                        )
                    )
                    select a.pk
                        , a.rang, a.fk_flstkennz_liemas
                        , a.datum as obsolet
                        --, b.rang, b.fk_flstkennz_liemas
                        , b.datum
                    from cte2 as a, cte2 as b
                    where 1
                    and a.rang > 1 and b.rang == 1
                    and substr(a.fk_flstkennz_liemas, 1,18) == substr(b.fk_flstkennz_liemas, 1,18)
                ) as a
                where st_alkis.pk == a.pk
                and st_alkis.gueltig_bis is NULL
                ;
                COMMIT;
            ''')

        print('Datumstempel "gueltig_bis" der betroffenen Flurstücke erfolgreich gesetzt')

    except sqlite3.Error as e:
        print('Fehlercode: ' + str(e))


    print('Starte abschließende Aufräumarbeiten:')

    try:
        with conn:
    
            # Löschen temporärer Flurstücksinformation
            _ = conn.execute(
                '''
                DELETE FROM t_alkis__wfs_wkt;
                '''
                )
    
        print('\tTemporäre Daten gelöscht')
    
    except sqlite3.Error as e:
        print('Fehlercode: ' + str(e))


    try:
        with conn:

            # Aktualisieren LayerStatistic
            _ = conn.execute(
                '''
                select updateLayerStatistics();
                '''
                )

        print('\tLayer-Statistik aktualisiert')

    except sqlite3.Error as e:
        print('Fehlercode: ' + str(e))


    # Verbindung schließen
    finally:
        conn.close()

    sleep(0.5)

    # Spatialite Treiber
    # conn = spatialite_connect(db_ac)
    conn = sqlite3.connect(db_ac)

    try:
        with conn:

            # Db-Bereinigung ()
            _ = conn.execute(
                '''
                vacuum;
                '''
                )

        print('\tDatenbank komprimiert')

    except sqlite3.Error as e:
        print('Fehlercode: ' + str(e))


    try:
        with conn:

            # Db-Bereinigung ()
            _ = conn.execute(
                '''
                pragma optimize;
                '''
                )

        print('\tDatenbank optimiert')

    except sqlite3.Error as e:
        print('Fehlercode: ' + str(e))

    # Verbindung schließen
    conn.close()


    # -------------------------------------------------------------------------

    globaltime_2 = datetime.now()
    # print('Prozess beendet nach : ' + str(globaltime_2-globaltime_1))
    
    # Aktualisierung Kartenansicht
    iface.mapCanvas().refreshAllLayers()

    show_message('Prozess beendet nach : '
                 + str(globaltime_2-globaltime_1), title="Download-Bestätigung")
